<?php
session_start();
if (!isset($_SESSION['username']) || $_SESSION['role'] != 'admin') {
    header("Location: login.php");
    exit();
}

include '../config/db.php';

// Initialize variables for totals
$total_quantity = 0;
$total_cost_price = 0;
$total_selling_price = 0;
$total_profit = 0;
$unique_drug_count = 0;

try {
    // Fetch the drugs data excluding specific drug names
    $stmt = $pdo->prepare("SELECT * FROM drugs WHERE category NOT IN ('service charge', 'service charges', 'consultation')");
    $stmt->execute();
    $drugs = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calculate totals and the number of unique drugs
    foreach ($drugs as $drug) {
        $total_quantity += $drug['quantity'];
        $total_cost_price += $drug['cost_price'] * $drug['quantity'];
        $total_selling_price += $drug['selling_price'] * $drug['quantity'];
        $total_profit += ($drug['selling_price'] - $drug['cost_price']) * $drug['quantity'];
    }

    // Get the unique number of drugs
    $unique_drug_count = count($drugs);

} catch (PDOException $e) {
    echo "Error: " . $e->getMessage();
    exit;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Stock Report - Store Management</title>
    <link rel="stylesheet" href="..\libs\bootsrtap\css\bootstrap.min.css">
    <link rel="stylesheet" href="..\libs\fontawesome\css\all.min.css">
    
    <style>
        body {
            background-color: #f5f6fa;
            font-family: 'Mulish', sans-serif;
            overflow-x: hidden;
        }

        /* Sidebar Styles */
        .sidebar {
            width: 250px;
            position: fixed;
            height: 100vh;
            background: #2c3e50;
            color: #ecf0f1;
            transition: all 0.3s ease;
            overflow-y: auto;
            z-index: 1000;
        }

        /* Content Styles */
        .content {
            margin-left: 250px;
            padding: 20px;
            transition: margin-left 0.3s ease;
        }

        /* Top Bar Styles */
        .top-bar {
            background-color: #ffffff;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        /* Cards Styles */
        .stats-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            transition: transform 0.3s ease;
        }

        .stats-card:hover {
            transform: translateY(-5px);
        }

        .stats-card h3 {
            color: #2c3e50;
            font-size: 1.1rem;
            margin-bottom: 15px;
        }

        .stats-card .value {
            font-size: 1.8rem;
            font-weight: bold;
            color: #3498db;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
            gap: 20px;
        }

        @media (max-width: 768px) {
            .sidebar {
                transform: translateX(-100%);
            }
            .sidebar.active {
                transform: translateX(0);
            }
            .content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>

<!-- Include Sidebar -->
<?php include '../includes/side_bar.php'; ?>

<div class="content">
    <!-- Top Bar -->
    <div class="top-bar">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h4 class="mb-0">Stock Report</h4>
                <small class="text-muted">Overview of current inventory</small>
            </div>
            <div class="d-flex gap-2">
                <button class="btn btn-outline-primary btn-sm d-md-none" onclick="toggleSidebar()">
                    <i class="fas fa-bars"></i>
                </button>
                <button class="btn btn-outline-primary btn-sm" onclick="window.print()">
                    <i class="fas fa-print"></i> Print Report
                </button>
            </div>
        </div>
    </div>

    <!-- Summary Cards -->
    <div class="stats-grid">
        <div class="stats-card">
            <h3><i class="fas fa-boxes me-2"></i>Total Products</h3>
            <p class="value"><?= $unique_drug_count ?></p>
            <small class="text-muted">Unique items in inventory</small>
        </div>

        <div class="stats-card">
            <h3><i class="fas fa-cubes me-2"></i>Total Quantity</h3>
            <p class="value"><?= number_format($total_quantity) ?></p>
            <small class="text-muted">Total items in stock</small>
        </div>

        <div class="stats-card">
            <h3><i class="fas fa-money-bill me-2"></i>Stock Value</h3>
            <p class="value">₦<?= number_format($total_cost_price, 2) ?></p>
            <small class="text-muted">Total inventory cost</small>
        </div>

        <div class="stats-card">
            <h3><i class="fas fa-chart-line me-2"></i>Potential Profit</h3>
            <p class="value">₦<?= number_format($total_profit, 2) ?></p>
            <small class="text-muted">Expected profit margin</small>
        </div>
    </div>

    <!-- Additional Details Card -->
    <div class="stats-card mt-4">
        <h3>Inventory Details</h3>
        <div class="table-responsive">
            <table class="table table-hover">
                <thead>
                    <tr>
                        <th>Category</th>
                        <th>Items Count</th>
                        <th>Total Value</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    // Group products by category
                    $categories = [];
                    foreach ($drugs as $drug) {
                        $cat = $drug['category'] ?? 'Uncategorized';
                        if (!isset($categories[$cat])) {
                            $categories[$cat] = [
                                'count' => 0,
                                'value' => 0
                            ];
                        }
                        $categories[$cat]['count']++;
                        $categories[$cat]['value'] += $drug['cost_price'] * $drug['quantity'];
                    }

                    foreach ($categories as $category => $data): ?>
                        <tr>
                            <td><?= htmlspecialchars($category) ?></td>
                            <td><?= $data['count'] ?></td>
                            <td>₦<?= number_format($data['value'], 2) ?></td>
                            <td>
                                <span class="badge bg-<?= $data['count'] > 0 ? 'success' : 'danger' ?>">
                                    <?= $data['count'] > 0 ? 'In Stock' : 'Out of Stock' ?>
                                </span>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script src="..\libs\jquery\jquery-3.7.1.min.js"></script>
<script src="..\libs\bootsrtap\js\bootstrap.bundle.min.js"></script>
<script>
function toggleSidebar() {
    document.querySelector('.sidebar').classList.toggle('active');
}
</script>

</body>
</html>
<script>
    function toggleSidebar() {
    document.querySelector('.sidebar').classList.toggle('active');
}
</script>