<?php
// Include database connection
include '../config/db.php'; // Adjust the path as necessary
session_start();
if (!isset($_SESSION['username']) || $_SESSION['role'] != 'admin') {
    header("Location: login.php");
    exit();
}


// Check user role
$user_role = $_SESSION['role']; // Assume user role is stored in the session

// Fetch all transactions with optional filtering
$filter = isset($_POST['filter']) ? htmlspecialchars($_POST['filter']) : '';
$query = "SELECT * FROM transactions";
if ($filter) {
    $query .= " WHERE username LIKE :filter OR activity_type LIKE :filter OR activity_details LIKE :filter";
}
$query .= " ORDER BY activity_date DESC"; // Sort by date descending
try {
    $stmt = $pdo->prepare($query);
    if ($filter) {
        $stmt->bindValue(':filter', "%$filter%");
    }
    $stmt->execute();
    $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die("Error fetching transactions: " . $e->getMessage());
}

// Redirect if the user does not have access
if ($user_role !== 'admin' && $user_role !== 'manager') {
    die("Access denied. You do not have permission to view this page.");
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>All Transactions - Store Management</title>
    <link rel="stylesheet" href="..\libs\bootsrtap\css\bootstrap.min.css">
    <link rel="stylesheet" href="..\libs\fontawesome\css\all.min.css">
    
    <style>
        body {
            background-color: #f5f6fa;
            font-family: 'Mulish', sans-serif;
            overflow-x: hidden;
        }

        .content {
            margin-left: 240px;
            padding: 20px;
            transition: margin-left 0.3s ease;
        }

        .top-bar {
            background-color: #ffffff;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            position: sticky;
            top: 0;
            z-index: 1000;
            backdrop-filter: blur(10px);
        }

        .transaction-card {
            background: white;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            padding: 20px;
            margin-bottom: 20px;
        }

        .table thead th {
            background-color: #2c3e50;
            color: white;
            font-weight: 500;
            border: none;
            white-space: nowrap;
        }

        .table td {
            vertical-align: middle;
        }

        .activity-badge {
            padding: 5px 10px;
            border-radius: 15px;
            font-size: 0.85rem;
            font-weight: 500;
        }

        .activity-login { background: #e8f5e9; color: #2e7d32; }
        .activity-logout { background: #fbe9e7; color: #d84315; }
        .activity-add { background: #e3f2fd; color: #1565c0; }
        .activity-update { background: #fff3e0; color: #ef6c00; }
        .activity-delete { background: #fce4ec; color: #c2185b; }

        .delete-btn {
            color: #e74c3c;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .delete-btn:hover {
            transform: scale(1.2);
        }

        @media (max-width: 768px) {
            .content {
                margin-left: 0;
                padding: 10px;
            }
            
            .mobile-toggle {
                display: block;
            }
        }
    </style>
</head>
<body>
    <?php include '../includes/side_bar.php'; ?>

    <!-- Mobile Toggle Button -->
    <button class="mobile-toggle d-md-none" onclick="toggleSidebar()">
        <i class="fas fa-bars"></i>
    </button>

    <div class="content">
        <!-- Top Bar -->
        <div class="top-bar">
            <div class="d-flex justify-content-between align-items-center">
                <h4 class="mb-0">Transaction History</h4>
                <form method="POST" class="d-flex gap-2">
                    <div class="input-group">
                        <input type="text" name="filter" class="form-control" 
                               placeholder="Search transactions..." 
                               value="<?= htmlspecialchars($filter) ?>">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-search"></i>
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Transactions Table -->
        <div class="transaction-card">
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>User</th>
                            <th>Activity</th>
                            <th>Details</th>
                            <th>Date & Time</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (count($transactions) > 0): ?>
                            <?php foreach ($transactions as $transaction): ?>
                                <tr>
                                    <td>#<?= htmlspecialchars($transaction['id']) ?></td>
                                    <td>
                                        <i class="fas fa-user-circle me-2"></i>
                                        <?= htmlspecialchars($transaction['username']) ?>
                                    </td>
                                    <td>
                                        <span class="activity-badge activity-<?= strtolower($transaction['activity_type']) ?>">
                                            <?= htmlspecialchars($transaction['activity_type']) ?>
                                        </span>
                                    </td>
                                    <td><?= htmlspecialchars($transaction['activity_details']) ?></td>
                                    <td>
                                        <i class="far fa-clock me-2"></i>
                                        <?= date('M d, Y H:i', strtotime($transaction['activity_date'])) ?>
                                    </td>
                                    <td>
                                        <i class="fas fa-trash delete-btn" 
                                           onclick="deleteTransaction(<?= $transaction['id'] ?>)"></i>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="6" class="text-center py-5">
                                    <i class="fas fa-info-circle mb-3 fa-2x text-muted"></i>
                                    <p class="text-muted">No transactions found.</p>
                                </td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <script src="..\libs\jquery\jquery-3.7.1.min.js"></script>
    <script src="..\libs\bootsrtap\js\bootstrap.bundle.min.js"></script>
    <script>
        function toggleSidebar() {
            document.querySelector('.sidebar').classList.toggle('active');
            document.querySelector('.content').classList.toggle('sidebar-active');
        }

        function deleteTransaction(id) {
            if (confirm('Are you sure you want to delete this transaction?')) {
                // Add your delete logic here
                console.log('Deleting transaction:', id);
            }
        }
    </script>
</body>
</html>
